<?php
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Start output buffering to prevent "headers already sent" errors
ob_start();

require_once __DIR__ . '/config/config.php';
require_once __DIR__ . '/includes/functions.php';
require_once __DIR__ . '/api/esim-api.php';

// Log session status
error_log("🔵 Checkout Topup - Session ID: " . session_id());
error_log("🔵 Checkout Topup - User ID: " . ($_SESSION['user_id'] ?? 'NOT SET'));
error_log("🔵 Checkout Topup - Request Method: " . $_SERVER['REQUEST_METHOD']);

requireLogin();

// Get parameters
$packageCode = isset($_GET['package']) ? sanitize($_GET['package']) : '';
$iccid = isset($_GET['iccid']) ? sanitize($_GET['iccid']) : '';

if (!$packageCode || !$iccid) {
    header('Location: /my-esims.php');
    exit;
}

// Get existing eSIM
$conn = getDBConnection();
$stmt = $conn->prepare("SELECT * FROM orders WHERE iccid = ? AND user_id = ? AND status = 'paid'");
$stmt->bind_param("si", $iccid, $_SESSION['user_id']);
$stmt->execute();
$result = $stmt->get_result();
$existingEsim = $result->fetch_assoc();
$stmt->close();

if (!$existingEsim || empty($existingEsim['iccid'])) {
    $_SESSION['error'] = 'eSIM مورد نظر یافت نشد';
    header('Location: /my-esims.php');
    exit;
}

$esimId = $existingEsim['id'];

// Get topup package details
$api = new eSIMAPI();
$topupResponse = $api->getTopupPackages($existingEsim['iccid'], $existingEsim['country_code']);

$package = null;
if ($topupResponse && isset($topupResponse['obj']['packageList'])) {
    foreach ($topupResponse['obj']['packageList'] as $pkg) {
        if ($pkg['packageCode'] == $packageCode && isset($pkg['supportTopUpType']) && $pkg['supportTopUpType'] >= 1) {
            $package = $pkg;
            break;
        }
    }
}

// If not found with ICCID, try without ICCID
if (!$package) {
    $topupResponse2 = $api->getTopupPackages('', $existingEsim['country_code']);
    if ($topupResponse2 && isset($topupResponse2['obj']['packageList'])) {
        foreach ($topupResponse2['obj']['packageList'] as $pkg) {
            if ($pkg['packageCode'] == $packageCode && isset($pkg['supportTopUpType']) && $pkg['supportTopUpType'] >= 1) {
                $package = $pkg;
                break;
            }
        }
    }
}

if (!$package) {
    $_SESSION['error'] = 'بسته شارژ مجدد یافت نشد';
    header('Location: /topup-esim.php?id=' . $esimId);
    exit;
}

// Calculate prices
$priceUsd = ($package['price'] ?? 0) / 10000;
$priceIrr = usdToIrr($priceUsd);

// Handle payment confirmation
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['confirm_topup'])) {
    error_log("🔵 Topup payment form submitted - eSIM ID: $esimId, Package: $packageCode");
    
    // Create topup order in database
    $stmt = $conn->prepare("INSERT INTO orders (user_id, package_code, package_name, country_code, country_name, price_usd, price_irr, status, order_type, parent_order_id) VALUES (?, ?, ?, ?, ?, ?, ?, 'pending', 'topup', ?)");
    
    $packageName = 'شارژ مجدد: ' . ($package['name'] ?? 'بسته');
    
    $stmt->bind_param("issssddi", 
        $_SESSION['user_id'],
        $packageCode,
        $packageName,
        $existingEsim['country_code'],
        $existingEsim['country_name'],
        $priceUsd,
        $priceIrr,
        $esimId
    );
    
    if ($stmt->execute()) {
        $orderId = $stmt->insert_id;
        $stmt->close();
        
        error_log("✅ Topup order created successfully - Order ID: $orderId, Parent eSIM: $esimId");
        error_log("🔄 Redirecting to: /proxy-payment.php?order_id=$orderId");
        
        // Redirect to payment
        header('Location: /proxy-payment.php?order_id=' . $orderId);
        exit;
    } else {
        $error = 'خطا در ثبت سفارش: ' . $stmt->error;
        error_log("❌ Failed to create topup order: " . $stmt->error);
        $stmt->close();
    }
}

$conn->close();

$pageTitle = 'تأیید شارژ مجدد eSIM';
include __DIR__ . '/includes/header.php';
?>

<style>
@import url('https://fonts.googleapis.com/css2?family=Vazirmatn:wght@400;500;600;700;800&display=swap');

body, * {
    font-family: 'Vazirmatn', Tahoma, Arial, sans-serif !important;
}

.fas, .far, .fab, .fa {
    font-family: "Font Awesome 6 Free", "Font Awesome 6 Brands" !important;
}

.container {
    max-width: 800px;
    margin: 0 auto;
    padding: 30px 20px;
}

.checkout-header {
    background: linear-gradient(135deg, #10b981 0%, #059669 100%);
    color: white;
    padding: 40px;
    border-radius: 20px;
    text-align: center;
    margin-bottom: 30px;
    box-shadow: 0 10px 30px rgba(16, 185, 129, 0.3);
}

.checkout-header h1 {
    font-size: 2.5rem;
    font-weight: 900;
    margin-bottom: 10px;
}

.info-box {
    background: white;
    padding: 30px;
    border-radius: 15px;
    margin-bottom: 25px;
    box-shadow: 0 4px 15px rgba(0,0,0,0.1);
}

.info-box h2 {
    color: #1e293b;
    font-size: 1.5rem;
    font-weight: 800;
    margin-bottom: 20px;
    display: flex;
    align-items: center;
    gap: 12px;
}

.info-box h2 i {
    color: #10b981;
}

.detail-row {
    display: flex;
    justify-content: space-between;
    padding: 15px 0;
    border-bottom: 1px solid #f1f5f9;
}

.detail-row:last-child {
    border-bottom: none;
}

.detail-row .label {
    color: #64748b;
    font-weight: 600;
}

.detail-row .value {
    color: #1e293b;
    font-weight: 800;
}

.price-box {
    background: linear-gradient(135deg, #10b981 0%, #059669 100%);
    color: white;
    padding: 30px;
    border-radius: 15px;
    text-align: center;
    margin-bottom: 25px;
}

.price-box .label {
    font-size: 1.2rem;
    margin-bottom: 10px;
    opacity: 0.95;
}

.price-box .amount {
    font-size: 3rem;
    font-weight: 900;
}

.alert {
    padding: 20px;
    border-radius: 12px;
    margin-bottom: 25px;
    display: flex;
    align-items: center;
    gap: 15px;
}

.alert-info {
    background: #e0e7ff;
    color: #3730a3;
    border: 2px solid #c7d2fe;
}

.alert-warning {
    background: #fef3c7;
    color: #92400e;
    border: 2px solid #fbbf24;
}

.esim-support-check {
    background: #fef3c7;
    border: 2px solid #fbbf24;
    border-radius: 12px;
    padding: 20px;
    margin-bottom: 20px;
}

.checkbox-container {
    display: flex;
    align-items: center;
    gap: 12px;
    cursor: pointer;
}

.checkbox-container input[type="checkbox"] {
    width: 24px;
    height: 24px;
    cursor: pointer;
}

.btn {
    width: 100%;
    padding: 18px;
    border: none;
    border-radius: 12px;
    font-size: 1.2rem;
    font-weight: 800;
    cursor: pointer;
    transition: all 0.3s ease;
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 12px;
}

.btn-success {
    background: linear-gradient(135deg, #10b981 0%, #059669 100%);
    color: white;
}

.btn-success:hover:not(:disabled) {
    transform: translateY(-2px);
    box-shadow: 0 5px 20px rgba(16, 185, 129, 0.4);
}

.btn:disabled {
    opacity: 0.5;
    cursor: not-allowed;
    background: #94a3b8 !important;
}

.btn-secondary {
    background: #e2e8f0;
    color: #475569;
    text-decoration: none;
    margin-top: 15px;
}

.btn-secondary:hover {
    background: #cbd5e1;
}

@media (max-width: 768px) {
    body {
        padding-top: 70px !important;
    }
    
    .container {
        padding: 15px 12px;
    }
    
    .checkout-header {
        padding: 25px 20px;
        border-radius: 16px;
    }
    
    .checkout-header h1 {
        font-size: 1.6rem;
    }
    
    .checkout-header p {
        font-size: 0.95rem;
    }
    
    .info-box {
        padding: 20px 18px;
    }
    
    .info-box h2 {
        font-size: 1.2rem;
    }
    
    .detail-row {
        flex-direction: column;
        gap: 8px;
        padding: 12px 0;
    }
    
    .detail-row .label {
        font-size: 0.9rem;
    }
    
    .detail-row .value {
        font-size: 0.95rem;
        word-break: break-all;
    }
    
    .price-box {
        padding: 25px 20px;
    }
    
    .price-box .label {
        font-size: 1rem;
    }
    
    .price-box .amount {
        font-size: 2.2rem;
    }
    
    .alert {
        padding: 16px;
        gap: 12px;
        font-size: 0.9rem;
    }
    
    .esim-support-check {
        padding: 16px;
    }
    
    .checkbox-container {
        font-size: 0.9rem;
    }
    
    .btn {
        padding: 16px;
        font-size: 1rem;
    }
}

@media (max-width: 480px) {
    .container {
        padding: 12px 10px;
    }
    
    .checkout-header {
        padding: 20px 16px;
    }
    
    .checkout-header h1 {
        font-size: 1.3rem;
    }
    
    .info-box {
        padding: 18px 15px;
    }
    
    .info-box h2 {
        font-size: 1.1rem;
    }
    
    .price-box .amount {
        font-size: 2rem;
    }
    
    .btn {
        padding: 14px;
        font-size: 0.95rem;
    }
}
</style>

<div class="container">
    <div class="checkout-header">
        <h1>
            <i class="fas fa-plus-circle"></i>
            تأیید شارژ مجدد eSIM
        </h1>
        <p>افزایش حجم و اعتبار به eSIM فعلی شما</p>
    </div>
    
    <?php if (isset($error)): ?>
        <div class="alert alert-warning">
            <i class="fas fa-exclamation-triangle"></i>
            <?php echo $error; ?>
        </div>
    <?php endif; ?>
    
    <div class="info-box">
        <h2>
            <i class="fas fa-sim-card"></i>
            eSIM فعلی
        </h2>
        <div class="detail-row">
            <span class="label">کشور:</span>
            <span class="value"><?php echo htmlspecialchars($existingEsim['country_name']); ?></span>
        </div>
        <div class="detail-row">
            <span class="label">بسته فعلی:</span>
            <span class="value"><?php echo htmlspecialchars($existingEsim['package_name']); ?></span>
        </div>
        <div class="detail-row">
            <span class="label">ICCID:</span>
            <span class="value" style="font-size: 0.9rem;"><?php echo htmlspecialchars($existingEsim['iccid']); ?></span>
        </div>
    </div>
    
    <div class="info-box">
        <h2>
            <i class="fas fa-box"></i>
            بسته شارژ مجدد
        </h2>
        <div class="detail-row">
            <span class="label">نام بسته:</span>
            <span class="value"><?php echo htmlspecialchars($package['name'] ?? 'بسته شارژ'); ?></span>
        </div>
        <div class="detail-row">
            <span class="label">حجم:</span>
            <span class="value"><?php echo round(($package['volume'] ?? 0) / 1073741824, 1); ?> GB</span>
        </div>
        <div class="detail-row">
            <span class="label">مدت:</span>
            <span class="value"><?php echo $package['duration'] ?? 0; ?> روز</span>
        </div>
    </div>
    
    <div class="alert alert-info">
        <i class="fas fa-info-circle"></i>
        <div>
            <strong>توجه:</strong> با پرداخت این مبلغ، حجم و مدت اعتبار به eSIM فعلی شما اضافه می‌شود و نیازی به نصب eSIM جدید نیست.
        </div>
    </div>
    
    <div class="price-box">
        <div class="label">مبلغ قابل پرداخت:</div>
        <div class="amount"><?php echo formatPrice($priceIrr); ?> تومان</div>
    </div>
    
    <form method="POST" id="topupForm" onsubmit="return validateForm()">
        <div class="esim-support-check">
            <label class="checkbox-container">
                <input type="checkbox" id="confirmTopup" name="confirm_checkbox" required>
                <span>
                    <i class="fas fa-check-circle"></i>
                    مطمئن هستم که می‌خواهم این بسته را به eSIM فعلی خود اضافه کنم
                </span>
            </label>
        </div>
        
        <button type="submit" name="confirm_topup" class="btn btn-success" id="paymentBtn" disabled>
            <i class="fas fa-credit-card"></i>
            پرداخت <?php echo formatPrice($priceIrr); ?> تومان
        </button>
        
        <!-- Debug info -->
        <div id="debugInfo" style="display: none; margin-top: 20px; padding: 15px; background: #f0f0f0; border-radius: 8px; font-size: 0.9rem;">
            <strong>🐛 اطلاعات دیباگ:</strong><br>
            Checkbox: <span id="checkboxStatus">❌</span><br>
            Button: <span id="buttonStatus">❌</span><br>
            Form: <span id="formStatus">آماده</span>
        </div>
        
        <div style="margin-top: 15px; text-align: center;">
            <a href="#" onclick="toggleDebug(); return false;" style="color: #999; font-size: 0.85rem; text-decoration: underline;">
                نمایش اطلاعات دیباگ
            </a>
        </div>
    </form>
    
    <a href="/topup-esim.php?id=<?php echo $esimId; ?>" class="btn btn-secondary">
        <i class="fas fa-arrow-right"></i>
        بازگشت
    </a>
</div>

<script>
// Debug mode
let debugMode = false;

function toggleDebug() {
    debugMode = !debugMode;
    const debugInfo = document.getElementById('debugInfo');
    if (debugInfo) {
        debugInfo.style.display = debugMode ? 'block' : 'none';
    }
    updateDebugInfo();
}

function updateDebugInfo() {
    if (!debugMode) return;
    
    const checkbox = document.getElementById('confirmTopup');
    const paymentBtn = document.getElementById('paymentBtn');
    
    document.getElementById('checkboxStatus').textContent = checkbox && checkbox.checked ? '✅ تیک خورده' : '❌ تیک نخورده';
    document.getElementById('buttonStatus').textContent = paymentBtn && !paymentBtn.disabled ? '✅ فعال' : '❌ غیرفعال';
}

function validateForm() {
    const checkbox = document.getElementById('confirmTopup');
    const formStatus = document.getElementById('formStatus');
    
    if (!checkbox.checked) {
        alert('لطفاً تیک تأیید را بزنید');
        if (formStatus) formStatus.textContent = '❌ تیک نخورده';
        return false;
    }
    
    if (formStatus) formStatus.textContent = '✅ در حال ارسال...';
    
    console.log('✅ Form is being submitted!');
    console.log('Checkbox checked:', checkbox.checked);
    console.log('Form action:', document.getElementById('topupForm').action || 'Same page (POST)');
    
    return true;
}

document.addEventListener('DOMContentLoaded', function() {
    console.log('🚀 Checkout Topup page loaded');
    
    const checkbox = document.getElementById('confirmTopup');
    const paymentBtn = document.getElementById('paymentBtn');
    
    if (!checkbox) {
        console.error('❌ Checkbox not found!');
        return;
    }
    
    if (!paymentBtn) {
        console.error('❌ Payment button not found!');
        return;
    }
    
    console.log('✅ Checkbox and button found');
    
    // Enable/disable button based on checkbox
    checkbox.addEventListener('change', function() {
        paymentBtn.disabled = !this.checked;
        console.log('Checkbox changed:', this.checked);
        console.log('Button disabled:', paymentBtn.disabled);
        updateDebugInfo();
    });
    
    // Log form submission
    const form = document.getElementById('topupForm');
    if (form) {
        form.addEventListener('submit', function(e) {
            console.log('📤 Form submit event triggered');
            console.log('Checkbox checked:', checkbox.checked);
            
            if (!checkbox.checked) {
                console.error('❌ Checkbox not checked, preventing submit');
                e.preventDefault();
                alert('لطفاً تیک تأیید را بزنید');
                return false;
            }
            
            console.log('✅ Form validation passed, submitting...');
        });
    }
    
    console.log('✅ Event listeners attached');
});
</script>

<?php include __DIR__ . '/includes/footer.php'; ?>

